/*
 * Decompiled with CFR 0.152.
 */
package org.opensaml.xml.security;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.math.BigInteger;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.KeyException;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.CRLException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.security.interfaces.DSAParams;
import java.security.interfaces.DSAPrivateKey;
import java.security.interfaces.DSAPublicKey;
import java.security.interfaces.ECPrivateKey;
import java.security.interfaces.ECPublicKey;
import java.security.interfaces.RSAPrivateCrtKey;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.DSAPublicKeySpec;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.KeySpec;
import java.security.spec.RSAPublicKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Set;
import javax.crypto.KeyGenerator;
import javax.crypto.SecretKey;
import org.apache.commons.ssl.PKCS8Key;
import org.apache.xml.security.Init;
import org.apache.xml.security.algorithms.JCEMapper;
import org.opensaml.xml.Configuration;
import org.opensaml.xml.encryption.EncryptionParameters;
import org.opensaml.xml.encryption.KeyEncryptionParameters;
import org.opensaml.xml.security.SecurityConfiguration;
import org.opensaml.xml.security.SecurityException;
import org.opensaml.xml.security.SigningUtil;
import org.opensaml.xml.security.credential.BasicCredential;
import org.opensaml.xml.security.credential.Credential;
import org.opensaml.xml.security.keyinfo.BasicProviderKeyInfoCredentialResolver;
import org.opensaml.xml.security.keyinfo.KeyInfoCredentialResolver;
import org.opensaml.xml.security.keyinfo.KeyInfoGenerator;
import org.opensaml.xml.security.keyinfo.KeyInfoGeneratorFactory;
import org.opensaml.xml.security.keyinfo.KeyInfoProvider;
import org.opensaml.xml.security.keyinfo.NamedKeyInfoGeneratorManager;
import org.opensaml.xml.security.keyinfo.provider.DEREncodedKeyValueProvider;
import org.opensaml.xml.security.keyinfo.provider.DSAKeyValueProvider;
import org.opensaml.xml.security.keyinfo.provider.InlineX509DataProvider;
import org.opensaml.xml.security.keyinfo.provider.RSAKeyValueProvider;
import org.opensaml.xml.security.x509.BasicX509Credential;
import org.opensaml.xml.signature.KeyInfo;
import org.opensaml.xml.signature.Signature;
import org.opensaml.xml.util.Base64;
import org.opensaml.xml.util.DatatypeHelper;
import org.opensaml.xml.util.LazySet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class SecurityHelper {
    private static Set<String> rsaAlgorithmURIs;
    private static Set<String> dsaAlgorithmURIs;
    private static Set<String> ecdsaAlgorithmURIs;

    private SecurityHelper() {
    }

    public static String getAlgorithmIDFromURI(String algorithmURI) {
        return DatatypeHelper.safeTrimOrNullString(JCEMapper.translateURItoJCEID((String)algorithmURI));
    }

    public static boolean isHMAC(String signatureAlgorithm) {
        String algoClass = DatatypeHelper.safeTrimOrNullString(JCEMapper.getAlgorithmClassFromURI((String)signatureAlgorithm));
        return "Mac".equals(algoClass);
    }

    public static String getKeyAlgorithmFromURI(String algorithmURI) {
        String apacheValue = DatatypeHelper.safeTrimOrNullString(JCEMapper.getJCEKeyAlgorithmFromURI((String)algorithmURI));
        if (apacheValue != null) {
            return apacheValue;
        }
        if (SecurityHelper.isHMAC(algorithmURI)) {
            return null;
        }
        if (rsaAlgorithmURIs.contains(algorithmURI)) {
            return "RSA";
        }
        if (dsaAlgorithmURIs.contains(algorithmURI)) {
            return "DSA";
        }
        if (ecdsaAlgorithmURIs.contains(algorithmURI)) {
            return "EC";
        }
        return null;
    }

    public static Integer getKeyLengthFromURI(String algorithmURI) {
        Logger log = SecurityHelper.getLogger();
        String algoClass = DatatypeHelper.safeTrimOrNullString(JCEMapper.getAlgorithmClassFromURI((String)algorithmURI));
        if ("BlockEncryption".equals(algoClass) || "SymmetricKeyWrap".equals(algoClass)) {
            try {
                int keyLength = JCEMapper.getKeyLengthFromURI((String)algorithmURI);
                return new Integer(keyLength);
            }
            catch (NumberFormatException e) {
                log.warn("XML Security config contained invalid key length value for algorithm URI: " + algorithmURI);
            }
        }
        log.info("Mapping from algorithm URI {} to key length not available", (Object)algorithmURI);
        return null;
    }

    public static SecretKey generateSymmetricKey(String algoURI) throws NoSuchAlgorithmException, KeyException {
        Logger log = SecurityHelper.getLogger();
        String jceAlgorithmName = SecurityHelper.getKeyAlgorithmFromURI(algoURI);
        if (DatatypeHelper.isEmpty(jceAlgorithmName)) {
            log.error("Mapping from algorithm URI '" + algoURI + "' to key algorithm not available, key generation failed");
            throw new NoSuchAlgorithmException("Algorithm URI'" + algoURI + "' is invalid for key generation");
        }
        Integer keyLength = null;
        keyLength = "http://www.w3.org/2001/04/xmlenc#tripledes-cbc".equals(algoURI) || "http://www.w3.org/2001/04/xmlenc#kw-tripledes".equals(algoURI) ? Integer.valueOf(168) : SecurityHelper.getKeyLengthFromURI(algoURI);
        if (keyLength == null) {
            log.error("Key length could not be determined from algorithm URI, can't generate key");
            throw new KeyException("Key length not determinable from algorithm URI, could not generate new key");
        }
        KeyGenerator keyGenerator = KeyGenerator.getInstance(jceAlgorithmName);
        keyGenerator.init(keyLength);
        return keyGenerator.generateKey();
    }

    public static Key extractEncryptionKey(Credential credential) {
        if (credential == null) {
            return null;
        }
        if (credential.getPublicKey() != null) {
            return credential.getPublicKey();
        }
        return credential.getSecretKey();
    }

    public static Key extractDecryptionKey(Credential credential) {
        if (credential == null) {
            return null;
        }
        if (credential.getPrivateKey() != null) {
            return credential.getPrivateKey();
        }
        return credential.getSecretKey();
    }

    public static Key extractSigningKey(Credential credential) {
        if (credential == null) {
            return null;
        }
        if (credential.getPrivateKey() != null) {
            return credential.getPrivateKey();
        }
        return credential.getSecretKey();
    }

    public static Key extractVerificationKey(Credential credential) {
        if (credential == null) {
            return null;
        }
        if (credential.getPublicKey() != null) {
            return credential.getPublicKey();
        }
        return credential.getSecretKey();
    }

    public static Integer getKeyLength(Key key) {
        Logger log = SecurityHelper.getLogger();
        if (key instanceof SecretKey && "RAW".equals(key.getFormat())) {
            return key.getEncoded().length * 8;
        }
        log.debug("Unable to determine length in bits of specified Key instance");
        return null;
    }

    public static BasicCredential getSimpleCredential(SecretKey secretKey) {
        if (secretKey == null) {
            throw new IllegalArgumentException("A secret key is required");
        }
        BasicCredential cred = new BasicCredential();
        cred.setSecretKey(secretKey);
        return cred;
    }

    public static BasicCredential getSimpleCredential(PublicKey publicKey, PrivateKey privateKey) {
        if (publicKey == null) {
            throw new IllegalArgumentException("A public key is required");
        }
        BasicCredential cred = new BasicCredential();
        cred.setPublicKey(publicKey);
        cred.setPrivateKey(privateKey);
        return cred;
    }

    public static BasicX509Credential getSimpleCredential(X509Certificate cert, PrivateKey privateKey) {
        if (cert == null) {
            throw new IllegalArgumentException("A certificate is required");
        }
        BasicX509Credential cred = new BasicX509Credential();
        cred.setEntityCertificate(cert);
        cred.setPrivateKey(privateKey);
        return cred;
    }

    public static SecretKey decodeSecretKey(byte[] key, char[] password) throws KeyException {
        throw new UnsupportedOperationException("This method is not yet supported");
    }

    public static PublicKey decodePublicKey(byte[] key, char[] password) throws KeyException {
        X509EncodedKeySpec keySpec = new X509EncodedKeySpec(key);
        try {
            return SecurityHelper.buildKey(keySpec, "RSA");
        }
        catch (KeyException keyException) {
            try {
                return SecurityHelper.buildKey(keySpec, "DSA");
            }
            catch (KeyException keyException2) {
                try {
                    return SecurityHelper.buildKey(keySpec, "EC");
                }
                catch (KeyException keyException3) {
                    throw new KeyException("Unsupported key type.");
                }
            }
        }
    }

    public static PublicKey derivePublicKey(PrivateKey key) throws KeyException {
        if (key instanceof DSAPrivateKey) {
            DSAPrivateKey dsaKey = (DSAPrivateKey)key;
            DSAParams keyParams = dsaKey.getParams();
            BigInteger y = keyParams.getG().modPow(dsaKey.getX(), keyParams.getP());
            DSAPublicKeySpec pubKeySpec = new DSAPublicKeySpec(y, keyParams.getP(), keyParams.getQ(), keyParams.getG());
            try {
                KeyFactory factory = KeyFactory.getInstance("DSA");
                return factory.generatePublic(pubKeySpec);
            }
            catch (GeneralSecurityException e) {
                throw new KeyException("Unable to derive public key from DSA private key", e);
            }
        }
        if (key instanceof RSAPrivateCrtKey) {
            RSAPrivateCrtKey rsaKey = (RSAPrivateCrtKey)key;
            RSAPublicKeySpec pubKeySpec = new RSAPublicKeySpec(rsaKey.getModulus(), rsaKey.getPublicExponent());
            try {
                KeyFactory factory = KeyFactory.getInstance("RSA");
                return factory.generatePublic(pubKeySpec);
            }
            catch (GeneralSecurityException e) {
                throw new KeyException("Unable to derive public key from RSA private key", e);
            }
        }
        throw new KeyException("Private key was not a DSA or RSA key");
    }

    public static PrivateKey decodePrivateKey(File key, char[] password) throws KeyException {
        if (!key.exists()) {
            throw new KeyException("Key file " + key.getAbsolutePath() + " does not exist");
        }
        if (!key.canRead()) {
            throw new KeyException("Key file " + key.getAbsolutePath() + " is not readable");
        }
        try {
            return SecurityHelper.decodePrivateKey(DatatypeHelper.fileToByteArray(key), password);
        }
        catch (IOException e) {
            throw new KeyException("Error reading Key file " + key.getAbsolutePath(), e);
        }
    }

    public static PrivateKey decodePrivateKey(byte[] key, char[] password) throws KeyException {
        try {
            PKCS8Key deocodedKey = new PKCS8Key(key, password);
            return deocodedKey.getPrivateKey();
        }
        catch (GeneralSecurityException e) {
            throw new KeyException("Unable to decode private key", e);
        }
    }

    public static X509Certificate buildJavaX509Cert(String base64Cert) throws CertificateException {
        CertificateFactory cf = CertificateFactory.getInstance("X.509");
        ByteArrayInputStream input = new ByteArrayInputStream(Base64.decode(base64Cert));
        return (X509Certificate)cf.generateCertificate(input);
    }

    public static X509CRL buildJavaX509CRL(String base64CRL) throws CertificateException, CRLException {
        CertificateFactory cf = CertificateFactory.getInstance("X.509");
        ByteArrayInputStream input = new ByteArrayInputStream(Base64.decode(base64CRL));
        return (X509CRL)cf.generateCRL(input);
    }

    public static DSAPublicKey buildJavaDSAPublicKey(String base64EncodedKey) throws KeyException {
        X509EncodedKeySpec keySpec = new X509EncodedKeySpec(Base64.decode(base64EncodedKey));
        return (DSAPublicKey)SecurityHelper.buildKey(keySpec, "DSA");
    }

    public static RSAPublicKey buildJavaRSAPublicKey(String base64EncodedKey) throws KeyException {
        X509EncodedKeySpec keySpec = new X509EncodedKeySpec(Base64.decode(base64EncodedKey));
        return (RSAPublicKey)SecurityHelper.buildKey(keySpec, "RSA");
    }

    public static ECPublicKey buildJavaECPublicKey(String base64EncodedKey) throws KeyException {
        X509EncodedKeySpec keySpec = new X509EncodedKeySpec(Base64.decode(base64EncodedKey));
        return (ECPublicKey)SecurityHelper.buildKey(keySpec, "EC");
    }

    public static RSAPrivateKey buildJavaRSAPrivateKey(String base64EncodedKey) throws KeyException {
        PrivateKey key = SecurityHelper.buildJavaPrivateKey(base64EncodedKey);
        if (!(key instanceof RSAPrivateKey)) {
            throw new KeyException("Generated key was not an RSAPrivateKey instance");
        }
        return (RSAPrivateKey)key;
    }

    public static DSAPrivateKey buildJavaDSAPrivateKey(String base64EncodedKey) throws KeyException {
        PrivateKey key = SecurityHelper.buildJavaPrivateKey(base64EncodedKey);
        if (!(key instanceof DSAPrivateKey)) {
            throw new KeyException("Generated key was not a DSAPrivateKey instance");
        }
        return (DSAPrivateKey)key;
    }

    public static ECPrivateKey buildJavaECPrivateKey(String base64EncodedKey) throws KeyException {
        PrivateKey key = SecurityHelper.buildJavaPrivateKey(base64EncodedKey);
        if (!(key instanceof ECPrivateKey)) {
            throw new KeyException("Generated key was not an ECPrivateKey instance");
        }
        return (ECPrivateKey)key;
    }

    public static PrivateKey buildJavaPrivateKey(String base64EncodedKey) throws KeyException {
        return SecurityHelper.decodePrivateKey(Base64.decode(base64EncodedKey), null);
    }

    public static PublicKey buildKey(KeySpec keySpec, String keyAlgorithm) throws KeyException {
        try {
            KeyFactory keyFactory = KeyFactory.getInstance(keyAlgorithm);
            return keyFactory.generatePublic(keySpec);
        }
        catch (NoSuchAlgorithmException e) {
            throw new KeyException(keyAlgorithm + "algorithm is not supported by the JCE", e);
        }
        catch (InvalidKeySpecException e) {
            throw new KeyException("Invalid key information", e);
        }
    }

    public static SecretKey generateKeyFromURI(String algoURI) throws NoSuchAlgorithmException, NoSuchProviderException {
        String jceAlgorithmName = JCEMapper.getJCEKeyAlgorithmFromURI((String)algoURI);
        int keyLength = JCEMapper.getKeyLengthFromURI((String)algoURI);
        return SecurityHelper.generateKey(jceAlgorithmName, keyLength, null);
    }

    public static KeyPair generateKeyPairFromURI(String algoURI, int keyLength) throws NoSuchAlgorithmException, NoSuchProviderException {
        String jceAlgorithmName = JCEMapper.getJCEKeyAlgorithmFromURI((String)algoURI);
        return SecurityHelper.generateKeyPair(jceAlgorithmName, keyLength, null);
    }

    public static SecretKey generateKey(String algo, int keyLength, String provider) throws NoSuchAlgorithmException, NoSuchProviderException {
        SecretKey key = null;
        KeyGenerator keyGenerator = null;
        keyGenerator = provider != null ? KeyGenerator.getInstance(algo, provider) : KeyGenerator.getInstance(algo);
        keyGenerator.init(keyLength);
        key = keyGenerator.generateKey();
        return key;
    }

    public static KeyPair generateKeyPair(String algo, int keyLength, String provider) throws NoSuchAlgorithmException, NoSuchProviderException {
        KeyPairGenerator keyGenerator = null;
        keyGenerator = provider != null ? KeyPairGenerator.getInstance(algo, provider) : KeyPairGenerator.getInstance(algo);
        keyGenerator.initialize(keyLength);
        return keyGenerator.generateKeyPair();
    }

    public static Credential generateKeyAndCredential(String algorithmURI) throws NoSuchAlgorithmException, NoSuchProviderException {
        SecretKey key = SecurityHelper.generateKeyFromURI(algorithmURI);
        BasicCredential credential = new BasicCredential();
        credential.setSecretKey(key);
        return credential;
    }

    public static Credential generateKeyPairAndCredential(String algorithmURI, int keyLength, boolean includePrivate) throws NoSuchAlgorithmException, NoSuchProviderException {
        KeyPair keyPair = SecurityHelper.generateKeyPairFromURI(algorithmURI, keyLength);
        BasicCredential credential = new BasicCredential();
        credential.setPublicKey(keyPair.getPublic());
        if (includePrivate) {
            credential.setPrivateKey(keyPair.getPrivate());
        }
        return credential;
    }

    public static KeyInfoCredentialResolver buildBasicInlineKeyInfoResolver() {
        ArrayList<KeyInfoProvider> providers = new ArrayList<KeyInfoProvider>();
        providers.add(new RSAKeyValueProvider());
        providers.add(new DSAKeyValueProvider());
        providers.add(new DEREncodedKeyValueProvider());
        providers.add(new InlineX509DataProvider());
        return new BasicProviderKeyInfoCredentialResolver(providers);
    }

    public static boolean matchKeyPair(PublicKey pubKey, PrivateKey privKey) throws SecurityException {
        Logger log = SecurityHelper.getLogger();
        if (pubKey == null || privKey == null) {
            throw new SecurityException("Either public or private key was null");
        }
        SecurityConfiguration secConfig = Configuration.getGlobalSecurityConfiguration();
        if (secConfig == null) {
            throw new SecurityException("Global security configuration was null, could not resolve signing algorithm");
        }
        String algoURI = secConfig.getSignatureAlgorithmURI(privKey.getAlgorithm());
        if (algoURI == null) {
            throw new SecurityException("Can't determine algorithm URI from key algorithm: " + privKey.getAlgorithm());
        }
        String jcaAlgoID = SecurityHelper.getAlgorithmIDFromURI(algoURI);
        if (jcaAlgoID == null) {
            throw new SecurityException("Can't determine JCA algorithm ID from algorithm URI: " + algoURI);
        }
        if (log.isDebugEnabled()) {
            log.debug("Attempting to match key pair containing key algorithms public '{}' private '{}', using JCA signature algorithm '{}'", new Object[]{pubKey.getAlgorithm(), privKey.getAlgorithm(), jcaAlgoID});
        }
        byte[] data = "This is the data to sign".getBytes();
        byte[] signature = SigningUtil.sign(privKey, jcaAlgoID, data);
        return SigningUtil.verify(pubKey, jcaAlgoID, signature, data);
    }

    public static void prepareSignatureParams(Signature signature, Credential signingCredential, SecurityConfiguration config, String keyInfoGenName) throws SecurityException {
        Logger log = SecurityHelper.getLogger();
        SecurityConfiguration secConfig = config != null ? config : Configuration.getGlobalSecurityConfiguration();
        String signAlgo = signature.getSignatureAlgorithm();
        if (signAlgo == null) {
            signAlgo = secConfig.getSignatureAlgorithmURI(signingCredential);
            signature.setSignatureAlgorithm(signAlgo);
        }
        if (SecurityHelper.isHMAC(signAlgo) && signature.getHMACOutputLength() == null) {
            signature.setHMACOutputLength(secConfig.getSignatureHMACOutputLength());
        }
        if (signature.getCanonicalizationAlgorithm() == null) {
            signature.setCanonicalizationAlgorithm(secConfig.getSignatureCanonicalizationAlgorithm());
        }
        if (signature.getKeyInfo() == null) {
            KeyInfoGenerator kiGenerator = SecurityHelper.getKeyInfoGenerator(signingCredential, secConfig, keyInfoGenName);
            if (kiGenerator != null) {
                try {
                    KeyInfo keyInfo = kiGenerator.generate(signingCredential);
                    signature.setKeyInfo(keyInfo);
                }
                catch (SecurityException e) {
                    log.error("Error generating KeyInfo from credential", (Throwable)e);
                    throw e;
                }
            } else {
                log.info("No factory for named KeyInfoGenerator {} was found for credential type {}", (Object)keyInfoGenName, (Object)signingCredential.getCredentialType().getName());
                log.info("No KeyInfo will be generated for Signature");
            }
        }
    }

    public static EncryptionParameters buildDataEncryptionParams(Credential encryptionCredential, SecurityConfiguration config, String keyInfoGenName) {
        Logger log = SecurityHelper.getLogger();
        SecurityConfiguration secConfig = config != null ? config : Configuration.getGlobalSecurityConfiguration();
        EncryptionParameters encParams = new EncryptionParameters();
        encParams.setEncryptionCredential(encryptionCredential);
        if (encryptionCredential == null) {
            encParams.setAlgorithm(secConfig.getAutoGeneratedDataEncryptionKeyAlgorithmURI());
        } else {
            encParams.setAlgorithm(secConfig.getDataEncryptionAlgorithmURI(encryptionCredential));
            KeyInfoGenerator kiGenerator = SecurityHelper.getKeyInfoGenerator(encryptionCredential, secConfig, keyInfoGenName);
            if (kiGenerator != null) {
                encParams.setKeyInfoGenerator(kiGenerator);
            } else {
                log.info("No factory for named KeyInfoGenerator {} was found for credential type{}", (Object)keyInfoGenName, (Object)encryptionCredential.getCredentialType().getName());
                log.info("No KeyInfo will be generated for EncryptedData");
            }
        }
        return encParams;
    }

    public static KeyEncryptionParameters buildKeyEncryptionParams(Credential encryptionCredential, String wrappedKeyAlgorithm, SecurityConfiguration config, String keyInfoGenName, String recipient) throws SecurityException {
        Logger log = SecurityHelper.getLogger();
        SecurityConfiguration secConfig = config != null ? config : Configuration.getGlobalSecurityConfiguration();
        KeyEncryptionParameters kekParams = new KeyEncryptionParameters();
        kekParams.setEncryptionCredential(encryptionCredential);
        if (encryptionCredential == null) {
            throw new SecurityException("Key encryption credential may not be null");
        }
        kekParams.setAlgorithm(secConfig.getKeyTransportEncryptionAlgorithmURI(encryptionCredential, wrappedKeyAlgorithm));
        KeyInfoGenerator kiGenerator = SecurityHelper.getKeyInfoGenerator(encryptionCredential, secConfig, keyInfoGenName);
        if (kiGenerator != null) {
            kekParams.setKeyInfoGenerator(kiGenerator);
        } else {
            log.info("No factory for named KeyInfoGenerator {} was found for credential type {}", (Object)keyInfoGenName, (Object)encryptionCredential.getCredentialType().getName());
            log.info("No KeyInfo will be generated for EncryptedKey");
        }
        kekParams.setRecipient(recipient);
        return kekParams;
    }

    public static KeyInfoGenerator getKeyInfoGenerator(Credential credential, SecurityConfiguration config, String keyInfoGenName) {
        SecurityConfiguration secConfig = config != null ? config : Configuration.getGlobalSecurityConfiguration();
        NamedKeyInfoGeneratorManager kiMgr = secConfig.getKeyInfoGeneratorManager();
        if (kiMgr != null) {
            KeyInfoGeneratorFactory kiFactory = null;
            kiFactory = DatatypeHelper.isEmpty(keyInfoGenName) ? kiMgr.getDefaultManager().getFactory(credential) : kiMgr.getFactory(keyInfoGenName, credential);
            if (kiFactory != null) {
                return kiFactory.newInstance();
            }
        }
        return null;
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(SecurityHelper.class);
    }

    static {
        if (!Init.isInitialized()) {
            Init.init();
        }
        dsaAlgorithmURIs = new LazySet<String>();
        dsaAlgorithmURIs.add("http://www.w3.org/2000/09/xmldsig#dsa-sha1");
        ecdsaAlgorithmURIs = new LazySet<String>();
        ecdsaAlgorithmURIs.add("http://www.w3.org/2001/04/xmldsig-more#ecdsa-sha1");
        ecdsaAlgorithmURIs.add("http://www.w3.org/2001/04/xmldsig-more#ecdsa-sha256");
        ecdsaAlgorithmURIs.add("http://www.w3.org/2001/04/xmldsig-more#ecdsa-sha384");
        ecdsaAlgorithmURIs.add("http://www.w3.org/2001/04/xmldsig-more#ecdsa-sha512");
        rsaAlgorithmURIs = new HashSet<String>(10);
        rsaAlgorithmURIs.add("http://www.w3.org/2000/09/xmldsig#rsa-sha1");
        rsaAlgorithmURIs.add("http://www.w3.org/2001/04/xmldsig-more#rsa-sha256");
        rsaAlgorithmURIs.add("http://www.w3.org/2001/04/xmldsig-more#rsa-sha384");
        rsaAlgorithmURIs.add("http://www.w3.org/2001/04/xmldsig-more#rsa-sha512");
        rsaAlgorithmURIs.add("http://www.w3.org/2001/04/xmldsig-more#rsa-sha512");
        rsaAlgorithmURIs.add("http://www.w3.org/2001/04/xmldsig-more#rsa-ripemd160");
        rsaAlgorithmURIs.add("http://www.w3.org/2001/04/xmldsig-more#rsa-md5");
    }
}

